module net.BurtonRadons.dig.platform.frame;

//private import std.gc; /+internal.gc.gc;+/ // removed by JCC

private import net.BurtonRadons.dig.platform.windows;
private import net.BurtonRadons.dig.platform.base;
private import net.BurtonRadons.dig.platform.control;
private import net.BurtonRadons.dig.platform.menu;
private import net.BurtonRadons.dig.platform.bitmap;

//private import std.c.windows.windows;
//import std.gc; /* Why is this needed? */
//private import std.c.windows.windows;

/** The base window class.
  * The window resizes according to the size and orientation of its
  * content, sizing so that there's ten pixels empty on each side.
  * Assigning the width and/or height directly stops this behaviour
  * for that axis.
  */
class Frame : Control
{
    private import std.c.stdlib;

    Dispatcher onClose;
    /**< Sent when the window or application should terminate.
       * When empty, the window is deleted if it's not the first-created
       * window; if it is, the application is exited with value 0.
       */

    Dispatcher onMaximized; /**< Sent when this window has been maximized. */
    Dispatcher onMinimized; /**< Sent when this window has been minimized. */

    Dispatcher onActive; /**< Window has been made current. */
    Dispatcher onInactive; /**< Another window has been made current. */

    Dispatcher onResized;
    /**< Window has been resized or restored after being minimized.
       * x and y are the dimensions of the new size asked for.
       */

    private static int digPlatformIndex = 0;

    private void digPlatformWindowClassCreate ()
    {
        digPlatformWindowClass.style = CS_HREDRAW | CS_VREDRAW;
        digPlatformWindowClass.lpfnWndProc = &digPlatformWindowProc;
        digPlatformWindowClass.cbClsExtra = 0;
        digPlatformWindowClass.cbWndExtra = 0;
        digPlatformWindowClass.hInstance = hinst;
        digPlatformWindowClass.hCursor = (_HANDLE) LoadCursorA ((_HANDLE) 0, IDC_ARROW);
        digPlatformWindowClass.hbrBackground = (_HBRUSH) ((_DWORD) digPlatformBackgroundColor + 1);
        digPlatformWindowClass.lpszMenuName = null;
        digPlatformWindowClass.lpszClassName = fmt ("frame%d", digPlatformIndex ++);
        std.c.windows.windows.RegisterClassA (&digPlatformWindowClass);
    }

    /** Setup the GUI class for windows. */
    static this ()
    {
        digPlatformBackgroundColor = (_HANDLE) COLOR_BTNFACE;
    }

    /** Create the window. */
    this (Control parent)
    {
        super (parent);
        digPlatformWindowClassCreate ();
        digPlatformStyle |= WS_OVERLAPPEDWINDOW;
        digPlatformHWNDCreate (0, toWStringz (digPlatformWindowClass.lpszClassName), null, digPlatformStyle, (_HANDLE) 0);
        if (digPlatformHWNDToControl.length == 1)
            digPlatformIsFirst = true;
    }

    /** Create the window with no parent (a top-level window). */
    this ()
    {
        this ((Control) null);
    }

    /** Create the window with a menu. */
    this (MenuBar menu)
    {
        super (null);

        digPlatformWindowClassCreate ();
        digPlatformMenuBar = menu;
        digPlatformStyle |= WS_OVERLAPPEDWINDOW;
        digPlatformHWNDCreate (0, toWStringz (digPlatformWindowClass.lpszClassName), null, digPlatformStyle, menu.digPlatformHandle);
        if (digPlatformHWNDToControl.length == 1)
            digPlatformIsFirst = true;
    }

    /** Get the background color. */
    override Color backgroundColor ()
    {
        uint color = GetSysColor ((int) digPlatformBackgroundColor);
        return AColor (color & 0xFF, (color >> 8) & 0xFF, (color >> 16) & 0xFF);
    }

    /** This version returns this. */
    override Control findFrame ()
    {
        return this;
    }

    /** Set the titlebar caption. */
    void caption (char [] text)
    {
        digPlatformSetCaption (text);
    }

    /** Bring this window to the front. */
    void bringToFront ()
    {
        BringWindowToTop (digPlatformHWND);
    }

    /** Display the window. */
    void display ()
    {
        digPlatformHidden = false;
        digPlatformGridfit ();
        if (!IsWindowVisible (digPlatformHWND))
            ShowWindow (digPlatformHWND, SW_NORMAL);
        //UpdateWindow (hwnd);
    }

    /** Hide the window so that it is not displayed. */
    void hide ()
    {
        ShowWindow (digPlatformHWND, SW_HIDE);
        std.c.windows.windows.UpdateWindow (digPlatformHWND);
        digPlatformHidden = true;
    }

    /** Close the window.  This is necessary if this is the modal window (if the showModal method has been called). */
    void close ()
    {
        digPlatformClosed = true;
        super.close ();
    }

private import std.gc;

    /** Receive and dispatch messages for the window. */
    void showModal ()
    {
        ulong ticks, lastKick = 0;
        std.c.windows.windows.MSG msg;
        int status;
        int pumped = 0;
        _HWND mhwnd;
        
        if (!digPlatformHidden)
        {
            display ();
            display ();
        }

        while ((status = std.c.windows.windows.GetMessageA (&msg, (_HANDLE) 0, 0, 0)) != 0 && !digPlatformClosed)
        {
            if (status == -1)
                return;

            std.c.windows.windows.TranslateMessage (&msg);

            if (msg.hwnd == (_HANDLE) 0 && msg.message == WM_TIMER)
            {
                KillTimer ((_HANDLE) 0, msg.wParam);
                for (int c; c < digPlatformTimers.length; c ++)
                    if (digPlatformTimers [c].mark == msg.wParam)
                    {
                        digPlatformTimers [c].dispatcher.notify ();
                        digPlatformTimers [c].mark = 0;
                        break;
                    }
            }

            std.c.windows.windows.DispatchMessageA (&msg);

            if ((ticks = GetTickCount ()) - lastKick > 10000)
            {
                lastKick = ticks;
                ulong a = GetTickCount ();
                //printf ("gc collecting\n");
                fullCollect (); //gc.fullCollect ();
                ulong b = GetTickCount ();
                //printf ("gc took %.2f seconds\n", (b - a) / 1000.0);
            }
        }

        delete this;
    }

    /** Set whether there should be a caption bar on the window. */
    void captionbar (bit value) { digPlatformSetRecreateStyle (WS_CAPTION, value); }

    /** Set whether the maximimize box should be displayed. */
    void maximizable (bit value) { digPlatformSetRecreateStyle (WS_MAXIMIZEBOX, value); }

    /** Set whether the minimize box should be displayed. */
    void minimizable (bit value) { digPlatformSetRecreateStyle (WS_MINIMIZEBOX, value); }

    /** Set whether this window is resizable. */
    void resizable (bit value) { digPlatformSetRecreateStyle (WS_SIZEBOX, value); }

    /** Set whether the system menu should be displayed. */
    void sysmenu (bit value) { digPlatformSetRecreateStyle (WS_SYSMENU, value); }

    /** Set whether this is a child window that runs in its
      * parent's frame (true) or an independent window that
      * runs at the top level (false, default).
      */

    void childWindow (bit value) { digPlatformSetRecreateStyle (WS_CHILD, value); }

    /** Quit the program. */
    static void quit ()
    {
        std.c.stdlib.exit (0);
        ExitProcess (0);
    }

    /** Assign the icon for the window.  Under Windows, the opacity channel is
      * used to create the transparency bitmap by making the pixel opaque if it's
      * closer to opacity than it is to transparency, or transparent otherwise.  The
      * default icon is system-selected, and can be reverted to by passing null
      * to this method.
      */

    void icon (Bitmap bitmap)
    {
        int w = GetSystemMetrics (SM_CXICON);
        int h = GetSystemMetrics (SM_CYICON);
    }

/+
#ifndef DOXYGEN_SHOULD_SKIP_THIS
+/

protected:
    bit digPlatformHidden = false;
    MenuBar digPlatformMenuBar;
    static _HBRUSH digPlatformBackgroundColor;
    _WNDCLASS digPlatformWindowClass;
    bit digPlatformIsFirst;

    bit digPlatformDoKeyDown (Control control, Event event)
    {
        if (control.bindings.notify (event, control.isFocus ()))
            return true;
        for (int c; c < control.childCount (); c ++)
            if (digPlatformDoKeyDown (control.child (c), event))
                return true;
        return false;
    }

    void digPlatformDoKeyDown (Event event)
    {
        digPlatformDoKeyDown (this, event);
    }

    static Control digPlatformPreviousMouseMove;

    static extern (Windows)
    _LRESULT digPlatformWindowProc (_HWND hwnd, _UINT message, _WPARAM wparam, _LPARAM lparam)
    {
        _HANDLE wparamHandle = (_HANDLE) wparam;
        Control control = digPlatformHWNDToControl [hwnd];
        Frame frame = cast (Frame) control;
        int result = 0;
        Event event;

        switch (message)
        {
            case WM_PAINT:
                if (frame === null || frame.onPaint.isEmpty ())
                    return std.c.windows.windows.DefWindowProcA (hwnd, message, wparam, lparam);
                frame.onPaint.notify ();
                std.c.windows.windows.ValidateRect (frame.digPlatformHWND, null);
                return 0;

            case WM_SETCURSOR:
            {
                Control cursorControl = digPlatformHWNDToControl [wparamHandle];

                if (cursorControl.digPlatformCursor)
                {
                    SetCursor (cursorControl.digPlatformCursor);
                    return 0;
                }
                else
                    return std.c.windows.windows.DefWindowProcA (hwnd, message, wparam, lparam);
            }

            case WM_CLOSE:
                if (!frame.onClose.isEmpty ())
                    frame.onClose.notify ();
                else if (frame.digPlatformIsFirst)
                {
                    delete frame;
                    ExitProcess (0);
                }
                else
                    delete frame;
                break;

            case WM_COMMAND:
                if (frame !== null && frame.digPlatformMenuBar !== null && lparam == 0)
                    frame.digPlatformMenuBar.digPlatformActivate (wparam & 0xFFFF);
                else
                {
                    Control commandControl = digPlatformHWNDToControl [(_HWND) lparam];

                    commandControl.digPlatformCommand (wparam >> 16, wparam & 0xFFFF);
                }
                break;

            case WM_LBUTTONDOWN:
                event.x = (short) (lparam & 0xFFFF);
                event.y = (short) (lparam >> 16);
                digPlatformEventKeySet (event, wparam);
                control.onLButtonDown.notify (event);
                break;

            case WM_LBUTTONUP:
                event.x = (short) (lparam & 0xFFFF);
                event.y = (short) (lparam >> 16);
                digPlatformEventKeySet (event, wparam);
                control.onLButtonUp.notify (event);
                break;

            case WM_MBUTTONDOWN:
                event.x = (short) (lparam & 0xFFFF);
                event.y = (short) (lparam >> 16);
                digPlatformEventKeySet (event, wparam);
                control.onMButtonDown.notify (event);
                break;

            case WM_MBUTTONUP:
                event.x = (short) (lparam & 0xFFFF);
                event.y = (short) (lparam >> 16);
                digPlatformEventKeySet (event, wparam);
                control.onMButtonUp.notify (event);
                break;

            case WM_RBUTTONDOWN:
                event.x = (short) (lparam & 0xFFFF);
                event.y = (short) (lparam >> 16);
                digPlatformEventKeySet (event, wparam);
                control.onRButtonDown.notify (event);
                break;

            case WM_RBUTTONUP:
                event.x = (short) (lparam & 0xFFFF);
                event.y = (short) (lparam >> 16);
                digPlatformEventKeySet (event, wparam);
                control.onRButtonUp.notify (event);
                break;

            case WM_MOUSEMOVE:
            {
                static int lastx, lasty;

                event.x = (short) (lparam & 0xFFFF);
                event.y = (short) (lparam >> 16);
                digPlatformEventKeySet (event, wparam);

                if (control !== digPlatformPreviousMouseMove)
                {
                    digPlatformPreviousMouseMove = control;
                    control.onMouseOver.notify (event);

                    TRACKMOUSEEVENT track;

                    track.cbSize = track.size;
                    track.dwFlags = TME_LEAVE;
                    track.hwndTrack = control.digPlatformHWND;
                    track.dwHoverTime = 0;
                    TrackMouseEvent (&track);
                }
                else
                {
                    event.deltax = lastx - event.x;
                    event.deltay = lasty - event.y;
                }

                lastx = event.x;
                lasty = event.y;
                control.onMouseMove.notify (event);
                return 0;
            }

            case WM_MOUSELEAVE:
                control.onMouseLeave.notify ();
                if (control === digPlatformPreviousMouseMove)
                    digPlatformPreviousMouseMove = null;
                break;

            case WM_MOUSEWHEEL:
                event.x = (short) (lparam & 0xFFFF);
                event.y = (short) (lparam >> 16);
                digPlatformEventKeySet (event, wparam);
                event.wheel = ((short) (wparam >> 16)) / (float) WHEEL_DELTA;
                control.onMouseWheel.notify (event);
                return 0;

            case WM_KEYDOWN:
                if (!digPlatformEventSetKeyCode (event, wparam))
                    break;
                event.keyRepeat = lparam & 0xFFFF;
                event.keyPrevious = (lparam & (1 << 30)) ? true : false;
                event.control = (GetKeyState (VK_CONTROL) & 0x8000) ? true : false;
                event.shift = (GetKeyState (VK_SHIFT) & 0x8000) ? true : false;
                
                if (control.findFrame () !== null)
                    (cast (Frame) control.findFrame ()).digPlatformDoKeyDown (event);
                control.onKeyDown.notify (event);
                break;

            case WM_KEYUP:
                if (!digPlatformEventSetKeyCode (event, wparam))
                    break;
                event.keyRepeat = lparam & 0xFFFF;
                event.keyPrevious = (lparam & (1 << 30)) ? true : false;
                event.control = (GetKeyState (VK_CONTROL) & 0x8000) ? true : false;
                event.shift = (GetKeyState (VK_SHIFT) & 0x8000) ? true : false;
                
                control.onKeyUp.notify (event);
                break;

            case WM_CHAR:
                event.keyCode = fmt ("%c", wparam);
                event.keyRepeat = lparam & 0xFFFF;
                event.keyPrevious = (lparam & (1 << 30)) ? true : false;
                control.onChar.notify (event);
                break;

            case WM_CTLCOLORBTN:
            {
                _HDC hdc = (_HDC) wparam;

                std.c.windows.windows.SetBkMode (hdc, TRANSPARENT);
                return (int) net.BurtonRadons.dig.platform.windows.CreateSolidBrush ((int) GetSysColor ((int) COLOR_BTNFACE));
            }

            case WM_CTLCOLORSTATIC:
            {
                _HDC hdc = (_HDC) wparam;

                std.c.windows.windows.SetBkMode (hdc, TRANSPARENT);
                return (int) CreateSolidBrush ((int) GetSysColor (COLOR_BTNFACE));
                return (_LRESULT) net.BurtonRadons.dig.platform.windows.CreateSolidBrush (digPlatformColorToCOLORREF ((cast (Frame) control).backgroundColor ()));
            }

            case WM_DROPFILES:
            {
                HDROP drop = (HDROP) wparam;
                char [4096] buffer;
                int count = DragQueryFileA (drop, uint.max, null, 0);
                char [] [] files;
                int length;
                _POINT point;

                //DragQueryPoint (drop, &point); //Joel Anderson (Needs to be added)

                for (int c; c < count; c ++)
                {
                    length = DragQueryFileA (drop, c, buffer, buffer.length);
                    files ~= buffer [0 .. length].dup;
                }

                DragFinish (drop);
                control.onDrop.notify (point.x, point.y, files);
                break;
            }

            case WM_KILLFOCUS:
                control.onLostFocus.notify (event);
                break;

            case WM_HELP:
            {
                HELPINFO *info = (HELPINFO *) lparam;

                control = digPlatformHWNDToControl [info.hItemHandle];
                while (control !== null)
                {
                    if (control.onHelp.methods.length)
                    {
                        control.onHelp.notify ();
                        break;
                    }
                    else
                        control = control.parent ();
                }
                break;
            }

            case WM_VSCROLL:
            {
                int page = control.vscrollPage ();
                int max = control.vscrollRangeMax ();
                int min = control.vscrollRangeMin ();

                {
                    SCROLLINFO info;

                    info.cbSize = info.size;
                    info.fMask = SIF_TRACKPOS | SIF_RANGE;
                    GetScrollInfo (control.digPlatformHWND, SB_VERT, &info);
                    event.scrollPoint = info.nTrackPos;
                }

                switch (wparam & 0xFFFF)
                {
                    case SB_LINEUP:
                        event.scrollType = event.Scroll.LineUp;
                        event.scrollDest = imax (0, control.vscrollPoint () - 1);
                        break;

                    case SB_LINEDOWN:
                        event.scrollType = event.Scroll.LineDown;
                        event.scrollDest = imin (max - page + 1, control.vscrollPoint () + 1);
                        break;

                    case SB_PAGEUP:
                        event.scrollType = event.Scroll.PageUp;
                        event.scrollDest = imax (min, event.scrollPoint - imax (4, page * 0.75));
                        break;

                    case SB_PAGEDOWN:
                        event.scrollType = event.Scroll.PageDown;
                        event.scrollDest = imin (max - page + 1, control.vscrollPoint () + imax (4, page * 0.75));
                        break;

                    case SB_THUMBPOSITION:
                        event.scrollType = event.Scroll.Drop;
                        event.scrollDest = control.vscrollPoint ();
                        break;

                    case SB_THUMBTRACK:
                        event.scrollType = event.Scroll.Track;
                        event.scrollDest = event.scrollPoint;
                        break;

                    case SB_ENDSCROLL:
                        event.scrollType = event.Scroll.End;
                        event.scrollDest = control.vscrollPoint ();
                        break;

                    case SB_BOTTOM:
                        event.scrollType = event.Scroll.Bottom;
                        event.scrollDest = max - page;
                        break;

                    case SB_TOP:
                        event.scrollType = event.Scroll.Top;
                        event.scrollDest = 0;
                        break;
                }

                if (control.onVScroll.isEmpty ())
                {
                    if (event.scrollDest != control.vscrollPoint ())
                    {
                        control.vscrollPoint (event.scrollDest);
                        control.paint ();
                    }
                }
                else
                    control.onVScroll.notify (event);
                break;
            }

            case WM_HSCROLL:
            {
                int page = control.hscrollPage ();
                int max = control.hscrollRangeMax ();
                int min = control.hscrollRangeMin ();

                {
                    SCROLLINFO info;

                    info.cbSize = info.size;
                    info.fMask = SIF_TRACKPOS | SIF_RANGE;
                    GetScrollInfo (control.digPlatformHWND, SB_HORZ, &info);
                    event.scrollPoint = info.nTrackPos;
                }

                switch (wparam & 0xFFFF)
                {
                    case SB_LINEUP:
                        event.scrollType = event.Scroll.LineUp;
                        event.scrollDest = imax (0, control.hscrollPoint () - 1);
                        break;

                    case SB_LINEDOWN:
                        event.scrollType = event.Scroll.LineDown;
                        event.scrollDest = imin (max - page + 1, control.hscrollPoint () + 1);
                        break;

                    case SB_PAGEUP:
                        event.scrollType = event.Scroll.PageUp;
                        event.scrollDest = imax (min, event.scrollPoint - imax (4, page * 0.75));
                        break;

                    case SB_PAGEDOWN:
                        event.scrollType = event.Scroll.PageDown;
                        event.scrollDest = imin (max - page + 1, control.hscrollPoint () + imax (4, page * 0.75));
                        break;

                    case SB_THUMBPOSITION:
                        event.scrollType = event.Scroll.Drop;
                        event.scrollDest = control.hscrollPoint ();
                        break;

                    case SB_THUMBTRACK:
                        event.scrollType = event.Scroll.Track;
                        event.scrollDest = event.scrollPoint;
                        break;

                    case SB_ENDSCROLL:
                        event.scrollType = event.Scroll.End;
                        event.scrollDest = control.hscrollPoint ();
                        break;

                    case SB_BOTTOM:
                        event.scrollType = event.Scroll.Bottom;
                        event.scrollDest = max - page;
                        break;

                    case SB_TOP:
                        event.scrollType = event.Scroll.Top;
                        event.scrollDest = 0;
                        break;
                }

                if (control.onHScroll.isEmpty ())
                {
                    if (event.scrollDest != control.hscrollPoint ())
                    {
                        control.hscrollPoint (event.scrollDest);
                        control.paint ();
                    }
                }
                else
                    control.onHScroll.notify (event);
                break;
            }

            case WM_SIZE:
                if (frame === null)
                    return 0;
                event.x = lparam & 0xFFFF;
                event.y = lparam >> 16;
                switch (wparam)
                {
                    case SIZE_MAXHIDE: /* Someone else has become maximized, the bastard. */
                        frame.digPlatformVisible = false;
                        break;

                    case SIZE_MAXIMIZED:
                        frame.digPlatformVisible = true;
                        if (frame.onMaximized.isEmpty ())
                            frame.digPlatformResizeContents (event.x, event.y);
                        frame.onMaximized.notify (event);
                        frame.digPlatformSuggestWidth = event.x;
                        frame.digPlatformSuggestHeight = event.y;
                        break;

                    case SIZE_MINIMIZED:
                        frame.digPlatformVisible = false;
                        frame.onMinimized.notify (event);
                        break;

                    case SIZE_RESTORED:
                        frame.digPlatformVisible = true;
                        if (frame.onResized.isEmpty ())
                            frame.digPlatformResizeContents (event.x, event.y);
                        frame.onResized.notify (event);
                        frame.digPlatformSuggestWidth = event.x;
                        frame.digPlatformSuggestHeight = event.y;
                        return 0;
                } 

                return net.BurtonRadons.dig.platform.windows.DefWindowProcA (hwnd, message, wparam, lparam);

            case WM_ACTIVATE:
                if (frame === null)
                    return 0;
                if (wparam == WA_ACTIVE || wparam == WA_CLICKACTIVE)
                    frame.onActive.notify (event);
                else
                    frame.onInactive.notify (event);
                break;

            default:
                return DefWindowProcA (hwnd, message, wparam, lparam);
        }

        return result;
    }

    import std.random;

    bit digPlatformColResizable (int col)
    {
        for (int c; c < childCount (); c ++)
        {
            Control child = this.child (c);

            if (child.digPlatformCol == -1 || child.digPlatformCol > col || child.digPlatformCol + child.digPlatformColSpan <= col)
                continue;
            if (child.digPlatformStickyE || child.digPlatformStickyHC)
                return true;
        }

        return false;
    }

    bit digPlatformColResizable (int col, int row, int rowspan)
    {
        for (int c; c < childCount (); c ++)
        {
            Control child = this.child (c);

            if (child.digPlatformCol == -1 || child.digPlatformCol > col || child.digPlatformCol + child.digPlatformColSpan <= col)
                continue;
            if (child.digPlatformRow >= row + rowspan || child.digPlatformRow + child.digPlatformRowSpan <= row)
                continue;
            if (child.digPlatformStickyE || child.digPlatformStickyHC)
                return true;
        }

        return false;
    }

    int digPlatformColResizableCount (int row, int rowspan)
    {
        int colmax, rowmax;
        int count = 0;

        gridExtents (colmax, rowmax);
        for (int c; c < colmax; c ++)
            if (digPlatformColResizable (c, row, rowspan))
                count ++;

        return count;
    }

    int digPlatformColResizableCount ()
    {
        int colmax, rowmax;
        int count = 0;

        gridExtents (colmax, rowmax);
        for (int c; c < colmax; c ++)
            if (digPlatformColResizable (c))
                count ++;

        return count;
    }

    bit digPlatformRowResizable (int row)
    {
        for (int c; c < childCount (); c ++)
        {
            Control child = this.child (c);

            if (child.digPlatformRow == -1 || child.digPlatformRow > row || child.digPlatformRow + child.digPlatformRowSpan <= row)
                continue;
            if (child.digPlatformStickyS || child.digPlatformStickyVC)
                return true;
        }

        return false;
    }

    bit digPlatformRowResizable (int row, int col, int colspan)
    {
        for (int c; c < childCount (); c ++)
        {
            Control child = this.child (c);

            if (child.digPlatformRow == -1 || child.digPlatformRow > row || child.digPlatformRow + child.digPlatformRowSpan <= row)
                continue;
            if (child.digPlatformCol >= col + colspan || child.digPlatformCol + child.digPlatformColSpan <= col)
                continue;
            if (child.digPlatformStickyS || child.digPlatformStickyVC)
                return true;
        }

        return false;
    }

    int digPlatformRowResizableCount ()
    {
        int colmax, rowmax;
        int count = 0;

        gridExtents (colmax, rowmax);
        for (int c; c < rowmax; c ++)
            if (digPlatformRowResizable (c))
                count ++;

        return count;
    }

    int digPlatformRowResizableCount (int col, int colspan)
    {
        int colmax, rowmax;
        int count = 0;

        gridExtents (colmax, rowmax);
        for (int c; c < rowmax; c ++)
            if (digPlatformRowResizable (c, col, colspan))
                count ++;

        return count;
    }

    bit digPlatformColDependent (Control child, Control other)
    {
        if (child === other)
            return false;
        if (child.digPlatformCol == -1 || other.digPlatformCol == -1)
            return false;
        if (other.digPlatformRow >= child.digPlatformRow + child.digPlatformRowSpan
         || other.digPlatformRow + other.digPlatformRowSpan <= child.digPlatformRow)
            return false;
        if (other.digPlatformCol < child.digPlatformCol)
            return false;
        return true;
    }

    bit digPlatformRowDependent (Control child, Control other)
    {
        if (child === other)
            return false;
        if (child.digPlatformCol == -1 || other.digPlatformCol == -1)
            return false;
        if (other.digPlatformCol >= child.digPlatformCol + child.digPlatformColSpan
         || other.digPlatformCol + other.digPlatformColSpan <= child.digPlatformCol)
            return false;
        if (other.digPlatformRow < child.digPlatformRow)
            return false;
        return true;
    }

    void digPlatformResizeContents (int nw, int nh)
    {
        int fw = this.width ();
        int fh = this.height ();
        int add, index;
        int oldx [] = new int [childCount ()];
        int oldy [] = new int [childCount ()];

        for (int c; c < childCount (); c ++)
        {
            Control child = this.child (c);
            int colchi, colnum = digPlatformColResizableCount (child.digPlatformCol, child.digPlatformColSpan);
            int rowchi, rownum = digPlatformRowResizableCount (child.digPlatformRow, child.digPlatformRowSpan);

            oldx [c] = child.width ();
            oldy [c] = child.height ();

            if (child.digPlatformCol == -1)
                continue;

            if (rownum) rowchi = (nh - fh) * child.digPlatformRowSpan / rownum;
            if (colnum) colchi = (nw - fw) * child.digPlatformColSpan / colnum;

            if (child.digPlatformStickyW && child.digPlatformStickyE)
                child.digPlatformSuggestWidth += colchi;
            else if (child.digPlatformStickyHC)
                child.digPlatformSuggestLeft += colchi / 2;
            else if (child.digPlatformStickyE)
                child.digPlatformSuggestLeft += colchi;
            else
                goto skipcols;

            for (int d; d < childCount (); d ++)
                if (digPlatformColDependent (child, this.child (d)))
                    this.child (d).digPlatformSuggestLeft += colchi;

        skipcols:
            if (child.digPlatformStickyN && child.digPlatformStickyS)
                child.digPlatformSuggestHeight += rowchi;
            else if (child.digPlatformStickyVC)
                child.digPlatformSuggestTop += rowchi;
            else if (child.digPlatformStickyS)
                child.digPlatformSuggestTop += rowchi;
            else
                continue;

            for (int d; d < childCount (); d ++)
                if (digPlatformRowDependent (child, this.child (d)))
                    this.child (d).digPlatformSuggestTop += rowchi;
        }

        for (int c; c < childCount (); c ++)
            this.child (c).digPlatformSetDimensions (oldx [c], oldy [c]);

        delete oldx;
        delete oldy;
    }

    override void digPlatformMoved ()
    {
        int rleft = left ();
        int rtop = top ();
        int rwidth = width ();
        int rheight = height ();
        _RECT wrect, crect;

        std.c.windows.windows.GetWindowRect (digPlatformHWND, &wrect);
        std.c.windows.windows.GetClientRect (digPlatformHWND, &crect);

        rleft = wrect.left;
        rtop = wrect.top;
        rwidth += (wrect.right - wrect.left) - (crect.right - crect.left);
        rheight += (wrect.bottom - wrect.top) - (crect.bottom - crect.top);
        MoveWindow (digPlatformHWND, rleft, rtop, rwidth, rheight, true);
        if (parent ())
            parent ().digPlatformChildMoved (this);
    }

    override void digPlatformChildMoved (Control control)
    {
        //gridfit ();
        //digPlatformMoved ();
    }

    override void digPlatformRecreate ()
    {
        if (digPlatformMenuBar !== null)
            throw new Error ("cannot recreate windows with menus (todo)");
        super.digPlatformRecreate ();
    }

/+
#endif
+/
};
